<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\User;
use App\Models\Rooms;
use App\Models\Booking;
use App\Models\RoomImage;
use App\Models\Apartments;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;

class RoomController extends Controller
{
    //
    use ApartmentLandlordViewController;
    public function showApartmentRooms(Apartments $apartment)
    {
        $data = $this->accessApartment($apartment);

    // Access the variables from the compact statement
    $apartment = $data['apartment'];
    $apartments = $data['apartments'];
    $statuses = ['Occupied', 'Booked', 'Available'];
        // Retrieve the rooms for the specified apartment
        $rooms = $apartment->rooms;
        $roomCount = $apartment->rooms->count();

        return view('landlord.apartment-rooms', compact('rooms', 'apartment','apartments','roomCount','statuses'));
    }
    public function assignApartment($roomId)
    {
        // Logic to assign an apartment to the room (e.g., update the room's apartment_id)
        $room = Rooms::find($roomId);
        $room->apartment_id = 1; // Specify the apartment ID as needed
        $room->save();

        return redirect()->route('rooms.index');
    }

    public function assignTenant($roomId, $userId)
    {
        // Logic to assign a tenant to the room (e.g., update the room's user_id)
        $room = Rooms::find($roomId);
        $userId= User::find($userId);
        $room->user_id = $userId; // Specify the user ID as needed
        $room->save();

        return redirect()->route('rooms.index');
    }

    public function removeTenant($roomId)
    {
        // Logic to remove the tenant from the room (e.g., set the room's user_id to null)
        $room = Rooms::find($roomId);
        $room->user_id = null;
        $room->save();

        return redirect()->route('rooms.index');
    }
    public function getRoomStatusOptions()
    {
        // Fetch room status options from the database
        $roomStatusOptions = Rooms::distinct('status')->pluck('status');

        return response()->json($roomStatusOptions);
    }
    public function getEmptyRoom(Request $request)
    {
        // Logic to retrieve rooms based on status
        $rooms = Rooms::where('status', 'empty')
                     ->orWhere('status', 'unoccupied')
                     ->orWhere('status', 'available')
                     ->orWhere('status', 'available_soon')
                     ->get();

        // Log the retrieved rooms
        \Log::info('Retrieved rooms: ' . $rooms);

        return response()->json(['rooms' => $rooms], 200);
    }

    public function getEmptyRoomWithLocation(Request $request)
    {
        // Retrieve the location parameter from the request
        $location = $request->input('location');

        // Retrieve empty rooms based on the apartment location
        $rooms = Rooms::where('status', 'empty')
                     ->whereHas('apartment', function ($query) use ($location) {
                         $query->where('location', $location);
                     })
                     ->get();

        return response()->json(['rooms' => $rooms], 200);
    }
    public function getEmptyRooms()
    {
        $endpoints = [
            'http://barakademo/api/v1/rooms',
            // 'https://api.example.com/rooms',
            // 'https://anotherapi.example.com/empty-rooms'
        ];

        $emptyRooms = [];

        foreach ($endpoints as $endpoint) {
            try {
                $response = Http::get($endpoint);

                if ($response->successful()) {
                    $rooms = $response->json();
                    $emptyRooms = array_merge($emptyRooms, $rooms);
                }
            } catch (\Exception $e) {
                // Handle exceptions or log errors
                // For now, we'll just ignore this endpoint if there's an error
            }
        }

        return response()->json($emptyRooms);
    }

    public function getPaymentStatusOptions()
    {
        // You can define payment status options statically or fetch them from the database
        $paymentStatusOptions = ['Paid', 'Unpaid', 'Cancelled'];

        return response()->json($paymentStatusOptions);
    }

    public function showGuestAvailableRooms(){
        return view('user.empty-rooms');
    }
    public function updateRoomAndPaymentStatus(Request $request)
    {
        // Validate and update the room and payment status in the database
        $user = auth()->user();

        // Example: Update the room status for the authenticated user
        $user->room->update(['status' => $request->input('roomStatus')]);

        // Example: Update the payment status for the authenticated user
        $user->update(['payment_status' => $request->input('paymentStatus')]);

        return response()->json(['success' => true, 'message' => 'Room and payment status updated successfully']);
    }
    public function book(Request $request)
    {
        // Validate the incoming request data
        $request->validate([
            'roomId' => 'required|exists:rooms,id', // Assuming your room ID field is named 'id' in the database
        ]);

        // Get the authenticated user
        $user = auth()->user();

        // Assuming you have a booking mechanism here, such as creating a booking record in the database
         $booking = Booking::create(['room_id' => $request->roomId, 'user_id' => $user->id]);

        // Return a response indicating success
        return response()->json(['message' => 'Room booked successfully', 'roomId' => $request->roomId]);
    }
    public function bookRoom(Request $request)
{
    // Validate the request data
    $request->validate([
        'room_id' => 'required|exists:rooms,id',
        'start_date' => 'required|date',
        'end_date' => 'required|date|after_or_equal:start_date',
    ]);

    // Create the booking record for the room
    $booking = $this->createBooking($request->room_id, 'App\Models\Room', $request->start_date, $request->end_date);

    // Return response
    return response()->json(['message' => 'Room booked successfully', 'booking' => $booking]);
}

public function bookService(Request $request)
{
    // Validate the request data
    $request->validate([
        'service_id' => 'required|exists:services,id',
        'start_date' => 'required|date',
        'end_date' => 'required|date|after_or_equal:start_date',
    ]);

    // Create the booking record for the service
    $booking = $this->createBooking($request->service_id, 'App\Models\Service', $request->start_date, $request->end_date);

    // Return response
    return response()->json(['message' => 'Service booked successfully', 'booking' => $booking]);
}

// Common function to create booking
private function createBooking($bookableId, $bookableType, $startDate, $endDate)
{
    // Calculate duration
    $startDate = Carbon::parse($startDate);
    $endDate = Carbon::parse($endDate);
    $duration = $startDate->diffInDays($endDate);

    // Create booking record
    $booking = new Booking();
    $booking->user_id = auth()->id();
    $booking->bookable_id = $bookableId;
    $booking->bookable_type = $bookableType;
    $booking->start_date = $startDate;
    $booking->end_date = $endDate;
    $booking->duration = $duration;
    $booking->status = 'pending';
    $booking->save();

    return $booking;
}

public function showRoomDetails($apartmentId, $roomId) {
    $data = $this->accessApartment($apartmentId);

    // Access the variables from the compact statement
    $apartment = $data['apartment'];
    $apartments = $data['apartments'];
    $statuses = ['Occupied', 'Booked', 'Available'];
        // Retrieve the rooms for the specified apartment
        $rooms = $apartment->rooms;

/// Find the room with the specified ID
$room = $rooms->firstWhere('id', $roomId);

// Check if the room exists
if (!$room) {
    // Handle the case where the room is not found
    abort(404); // Or any other appropriate action
}
    // Load images for the current room
    $room->load('images');


    // Retrieve the list of users who have booked the room
    // $bookedUsers = $room->bookings()->with('user')->get()->pluck('user');

    // Retrieve the list of users who have added the room to their watchlist
    // $watchlistUsers = $room->watchlistUsers()->get();

    // Display only the images for the current room
    $roomImages = $room->images;


    // Pass the room and room images to the view
    return view('landlord.room_details', compact('rooms', 'apartments', 'apartment', 'room'));
// End the loop


}


public function uploadImage(Request $request, Rooms $room)
    {
        // Validate the uploaded image
        $request->validate([
            'image' => 'required|image|mimes:jpeg,png,jpg,gif,webp|max:2048', // Adjust validation rules as needed
            'description' => 'nullable|string',
        ]);

        // Store the uploaded image in the 'room_images' directory within the 'public' disk
        $imagePath = $request->file('image')->store('room_images', 'public');



        // Create a new room image record in the database
        $roomImage = new RoomImage();
        $roomImage->room_id = $room->id;
        $roomImage->url = $imagePath;
        $roomImage->description = $request->description;
        $roomImage->save();

        // Redirect back to the room details page with a success message
        return redirect()->back()->with('success', 'Image uploaded successfully.');
    }

        public function updateRoomOrder(Request $request)
        {
            $apartmentId = $request->input('apartmentId');
            $roomOrder = $request->input('roomOrder');

            foreach ($roomOrder as $index => $roomId) {
                // Update the 'order' column in the rooms table for the specific apartment
                Rooms::where('id', $roomId)
                    ->where('apartment_id', $apartmentId)
                    ->update(['order' => $index + 1]);
            }

            return response()->json(['success' => true]);
        }
}
