<?php

namespace App\Models;

use Carbon\Carbon;
use App\Models\Rooms;
use App\Models\Apartments;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class License extends Model
{  protected $fillable = ['name', 'license_type', 'status', 'user_id', 'description', 'price','valid_until', 'total_apartments', 'room_id', 'total_rooms'];

    protected $dates = ['valid_until'];

    // Relationships
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function apartments()
    {
        return $this->hasMany(Apartments::class);
    }

    public function room()
    {
        return $this->belongsTo(Rooms::class);
    }

    // Custom methods

    /**
     * Check if the license is valid.
     *
     * @return bool
     */
    public function isValid()
    {
        return $this->valid_until && $this->valid_until->isFuture();
    }

    /**
     * Check if the license allows creating more apartments.
     *
     * @return bool
     */
    public function allowsApartmentCreation()
    {
        return $this->remainingApartments() > 0;
    }

    /**
     * Calculate the remaining apartments based on the license type.
     *
     * @return int
     */
    public function remainingApartments()
    {
        // Placeholder logic, adjust based on your license types and rules
        $occupiedApartments = $this->apartments()->count();
        return max(0, $this->total_apartments - $occupiedApartments);
    }

    /**
     * Calculate the remaining rooms based on the license type.
     *
     * @return int
     */
    public function remainingRooms()
    {
        // Placeholder logic, adjust based on your license types and rules
        $occupiedRooms = $this->apartments()->sum('rooms_count');
        return max(0, $this->total_rooms - $occupiedRooms);
    }

    /**
     * Calculate the license expiry date based on the type.
     *
     * @return Carbon|null
     */
    public function calculateExpiryDate()
    {
        // Placeholder logic, adjust based on your license types and rules
        $expiryMonths = 1; // Default expiry for monthly licenses

        if ($this->type === 'demo') {
            $expiryMonths = 6; // Demo license expires in 6 months
        } elseif ($this->type === 'standard') {
            $expiryMonths = 12; // Standard license expires in 12 months
        }

        return $this->created_at ? $this->created_at->addMonths($expiryMonths) : null;
    }

    /**
     * Create a new license for a user.
     *
     * @param string $type
     * @param int $totalApartments
     * @param int $totalRooms
     * @param int $roomId
     * @return License
     */
    public static function createLicense(User $user, $type = 'demo', $totalApartments = 1, $totalRooms = 1, $roomId = null)
    {
        return $user->licenses()->create([
            'type' => $type,
            'valid_until' => static::calculateExpiryDate(),
            'total_apartments' => $totalApartments,
            'room_id' => $roomId,
            'total_rooms' => $totalRooms,
        ]);
    }
    protected static function boot()
    {
        parent::boot();

        // Set a unique name with the prefix before creating a new license
        static::creating(function ($license) {
            $license->name = 'PACMIS-' . uniqid();
        });
    }
}
