<?php

namespace App\Models;


use Carbon\Carbon;
use Illuminate\Support\Str;
use App\Models\Rooms;
use App\Models\Apartments;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;


class LicensePurchase extends Model
{
    use HasFactory;
    protected $fillable = ['user_id', 'license_type', 'license_key', 'payment_status','payment_status', 'purchase_date', 'expiry_date'];

    // Relationships
    public function user()
    {
        return $this->belongsTo(User::class, 'license_type');
    }
    public function apartments()
    {
        return $this->hasOne(Apartments::class);
    }
    // Add the isValid method to check if a license purchase is valid
    public function isValid()
    {
        // Your logic to determine if the license purchase is valid
        // For example, check if the payment status is 'success' and other conditions
        return $this->payment_status === 'success' && $this->expires_at > now();
    }
    public function allowsApartmentCreation()
    {
        return $this->remainingApartments() > 0;
    }

    /**
     * Calculate the remaining apartments based on the license type.
     *
     * @return int
     */
    public function remainingApartments()
    {
        // Placeholder logic, adjust based on your license types and rules
        $occupiedApartments = $this->apartments()->count();
        return max(0, $this->total_apartments - $occupiedApartments);
    }

    /**
     * Calculate the remaining rooms based on the license type.
     *
     * @return int
     */
    public function remainingRooms()
    {
        // Placeholder logic, adjust based on your license types and rules
        $occupiedRooms = $this->apartments()->sum('rooms_count');
        return max(0, $this->total_rooms - $occupiedRooms);
    }

    /**
     * Calculate the license expiry date based on the type.
     *
     * @return Carbon|null
     */

    /**
     * Create a new license for a user.
     *
     * @param string $type
     * @param int $totalApartments
     * @param int $totalRooms
     * @param int $roomId
     * @return License
     */
    public static function createLicense(User $user, $type = 'demo', $totalApartments = 1, $totalRooms = 1, $roomId = null)
    {
        return $user->licenses()->create([
            'type' => $type,
            'valid_until' => static::calculateExpiryDate(),
            'total_apartments' => $totalApartments,
            'room_id' => $roomId,
            'total_rooms' => $totalRooms,
        ]);
    }
    public static function boot()
    {
        parent::boot();

        self::creating(function ($model) {
            $model->license_key = 'SA-' . Str::uuid();
        });

        self::saving(function ($model) {
            // Set purchase_date and calculate expiry_date based on the license type
            if ($model->isDirty('payment_status') && $model->payment_status === 'success') {
                $model->purchase_date = Carbon::now();
                $model->expiry_date = $model->calculateExpiryDate();
            }
        });
    }

    public function calculateExpiryDate()
    {
        $expiryPeriod = 0; // Default expiry period in months

        switch ($this->license_type) {
            case 'demo':
                $expiryPeriod = 1; // 1 week for demo
                break;
            case 'starter':
            case 'basic':
            case 'standard':
                $expiryPeriod = 1; // 1 month for starter, basic, standard
                break;
            case 'executive':
                $expiryPeriod = 12; // 1 year for executive
                break;
            case 'pro':
                $expiryPeriod = 6; // 6 months for pro
                break;
            // Add more cases for other license types if needed
        }

        return Carbon::parse($this->purchase_date)->addMonths($expiryPeriod);
    }

    public function renewLicense($months)
    {
        // Validate that the license is eligible for renewal
        if ($this->payment_status === 'success' && $this->expiry_date && $this->expiry_date > Carbon::now()) {
            // Calculate the new expiry date based on the selected months
            $newExpiryDate = Carbon::parse($this->expiry_date)->addMonths($months);
            $this->expiry_date = $newExpiryDate;
            $this->save();

            return true; // Renewal successful
        }

        return false; // Renewal not allowed or failed
    }

    public function licensesInvoices()
    {
        return $this->hasMany(LicensesInvoice::class);
    }
}
